//                                               -*- C++ -*-
/**
 *  @brief OptimizationProblemImplementation allows one to describe an optimization problem
 *
 *  Copyright 2005-2025 Airbus-EDF-IMACS-ONERA-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#include <cstdlib>

#include "openturns/OptimizationProblemImplementation.hxx"
#include "openturns/PersistentObjectFactory.hxx"
#include "openturns/IdentityMatrix.hxx"
#include "openturns/QuadraticFunction.hxx"
#include "openturns/LinearFunction.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(OptimizationProblemImplementation)

static const Factory<OptimizationProblemImplementation> Factory_OptimizationProblemImplementation;

/* Default constructor */
OptimizationProblemImplementation::OptimizationProblemImplementation()
  : PersistentObject()
  , minimizationCollection_(1, true)
{
  // Nothing to do
}

OptimizationProblemImplementation::OptimizationProblemImplementation(const Function & objective)
  : PersistentObject()
  , objective_(objective)
  , minimizationCollection_(objective.getOutputDimension(), true)
  , dimension_(objective.getInputDimension())
  , variablesType_(dimension_, CONTINUOUS)
{
  // Nothing to do
}

/*
 * @brief General multi-objective equality, inequality and bound constraints
 */
OptimizationProblemImplementation::OptimizationProblemImplementation( const Function & objective,
    const Function & equalityConstraint,
    const Function & inequalityConstraint,
    const Interval & bounds)
  : PersistentObject()
  , objective_(objective)
  , minimizationCollection_(objective.getOutputDimension(), true)
  , dimension_(objective.getInputDimension())
  , variablesType_(Indices(dimension_, CONTINUOUS))
{
  // Set constraints
  setEqualityConstraint(equalityConstraint);
  setInequalityConstraint(inequalityConstraint);

  // Set bounds
  setBounds(bounds);
}


/* Virtual constructor */
OptimizationProblemImplementation * OptimizationProblemImplementation::clone() const
{
  return new OptimizationProblemImplementation(*this);
}

/* Objective accessor */
Function OptimizationProblemImplementation::getObjective() const
{
  return objective_;
}

void OptimizationProblemImplementation::setObjective(const Function & objective)
{
  if (objective.getInputDimension() != objective_.getInputDimension())
  {
    LOGINFO(OSS() << "Clearing constraints, bounds and variables types");
    // Clear constraints
    if (equalityConstraint_.getEvaluation().getImplementation()->isActualImplementation() || inequalityConstraint_.getEvaluation().getImplementation()->isActualImplementation())
    {
      equalityConstraint_ = Function();
      inequalityConstraint_ = Function();
    }

    // Clear bounds
    bounds_ = Interval(0);

    // Clear variables types
    variablesType_ = Indices(dimension_, CONTINUOUS);

  }
  objective_ = objective;

  // Update dimension_ member accordingly
  dimension_ = objective.getInputDimension();

  // Update variablesType_ member accordingly
  variablesType_ = Indices(dimension_, CONTINUOUS);
}

Bool OptimizationProblemImplementation::hasMultipleObjective() const
{
  return objective_.getOutputDimension() > 1;
}

/* Equality constraint accessor */
Function OptimizationProblemImplementation::getEqualityConstraint() const
{
  return equalityConstraint_;
}

void OptimizationProblemImplementation::setEqualityConstraint(const Function & equalityConstraint)
{
  if (!(equalityConstraint.getInputDimension() == 0 || equalityConstraint.getInputDimension() == dimension_))
    throw InvalidArgumentException(HERE) << "Error: the given equality constraints have an input dimension=" << equalityConstraint.getInputDimension() << " different from the input dimension=" << dimension_ << " of the objective.";

  equalityConstraint_ = equalityConstraint;
}

Bool OptimizationProblemImplementation::hasEqualityConstraint() const
{
  return equalityConstraint_.getEvaluation().getImplementation()->isActualImplementation();
}

/* Inequality constraint accessor */
Function OptimizationProblemImplementation::getInequalityConstraint() const
{
  return inequalityConstraint_;
}

void OptimizationProblemImplementation::setInequalityConstraint(const Function & inequalityConstraint)
{
  if (!(inequalityConstraint.getInputDimension() == 0 || inequalityConstraint.getInputDimension() == dimension_)) throw InvalidArgumentException(HERE) << "Error: the given inequality constraints have an input dimension=" << inequalityConstraint.getInputDimension() << " different from the input dimension=" << dimension_ << " of the objective.";

  inequalityConstraint_ = inequalityConstraint;
}

Bool OptimizationProblemImplementation::hasInequalityConstraint() const
{
  return inequalityConstraint_.getEvaluation().getImplementation()->isActualImplementation();
}

/* Bounds accessor */
Interval OptimizationProblemImplementation::getBounds() const
{
  return bounds_;
}

void OptimizationProblemImplementation::setBounds(const Interval & bounds)
{
  if (!(bounds.getDimension() == 0 || bounds.getDimension() == dimension_)) throw InvalidArgumentException(HERE) << "Error: the given bounds are of dimension=" << bounds.getDimension() << " different from the input dimension=" << dimension_ << " of the objective.";

  bounds_ = bounds;
}

Bool OptimizationProblemImplementation::hasBounds() const
{
  return bounds_.getDimension() > 0;
}

/* Level function accessor */
Function OptimizationProblemImplementation::getLevelFunction() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getLevelFunction";
}

void OptimizationProblemImplementation::setLevelFunction(const Function & /*levelFunction*/)
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::setLevelFunction";
}

Bool OptimizationProblemImplementation::hasLevelFunction() const
{
  return false;
}

Function OptimizationProblemImplementation::getResidualFunction() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getResidualFunction";
}

void OptimizationProblemImplementation::setResidualFunction(const Function & /*residualFunction*/)
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::setResidualFunction";
}

Bool OptimizationProblemImplementation::hasResidualFunction() const
{
  return false;
}

/* Level value accessor */
Scalar OptimizationProblemImplementation::getLevelValue() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getLevelValue";
}

void OptimizationProblemImplementation::setLevelValue(Scalar /*levelValue*/)
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::setLevelValue";
}

/* Linear flag accessor */
Bool OptimizationProblemImplementation::isLinear() const
{
  return false;
}

Point OptimizationProblemImplementation::getLinearCost() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getLinearCost";
}

Matrix OptimizationProblemImplementation::getLinearConstraintCoefficients() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getLinearConstraintCoefficients";
}

Interval OptimizationProblemImplementation::getLinearConstraintBounds() const
{
  throw NotYetImplementedException(HERE) << "in OptimizationProblemImplementation::getLinearConstraintBounds";
}

/* Dimension accessor */
UnsignedInteger OptimizationProblemImplementation::getDimension() const
{
  return dimension_;
}

/* Minimization accessor */
void OptimizationProblemImplementation::setMinimization(Bool minimization, UnsignedInteger marginalIndex)
{
  if (marginalIndex >= objective_.getOutputDimension())
    throw InvalidDimensionException(HERE) << "marginal index (" << marginalIndex << ") cannot exceed objective dimension (" << objective_.getOutputDimension();
  minimizationCollection_[marginalIndex] = minimization;
}

Bool OptimizationProblemImplementation::isMinimization(UnsignedInteger marginalIndex) const
{
  if (marginalIndex >= objective_.getOutputDimension())
    throw InvalidDimensionException(HERE) << "marginal index (" << marginalIndex << ") cannot exceed objective dimension (" << objective_.getOutputDimension();
  return minimizationCollection_[marginalIndex];
}

// Variables type table
void OptimizationProblemImplementation::setVariablesType(const Indices & variablesType)
{
  if (variablesType.getSize() != getDimension())
    throw InvalidDimensionException(HERE) << "variables type table dimension is invalid (" << variablesType.getSize() << ", expected " << getDimension();

  variablesType_ = variablesType;
}

Indices OptimizationProblemImplementation::getVariablesType() const
{
  return variablesType_;
}

Bool OptimizationProblemImplementation::isContinuous() const
{
  if (dimension_ == 0)
    return true;

  for (UnsignedInteger i = 0; i < dimension_; ++i)
    if (variablesType_[i] != CONTINUOUS) return false;

  return true;
}

/* String converter */
String OptimizationProblemImplementation::__repr__() const
{
  OSS oss;
  oss << "class=" << OptimizationProblemImplementation::GetClassName();
  oss << " objective=" << objective_
      << " equality constraint=" << (hasEqualityConstraint() ? equalityConstraint_.__repr__() : "none")
      << " inequality constraint=" << (hasInequalityConstraint() ? inequalityConstraint_.__repr__() : "none");
  oss << " bounds=" << (hasBounds() ? bounds_.__repr__() : "none");
  if (minimizationCollection_.getSize() == 1)
    oss << " minimization=" << (minimizationCollection_[0] ? true : false);
  else
    oss << " minimization=" << minimizationCollection_;
  oss << " dimension=" << dimension_;
  return oss;
}

/* Method save() stores the object through the StorageManager */
void OptimizationProblemImplementation::save(Advocate & adv) const
{
  PersistentObject::save(adv);
  adv.saveAttribute( "objective_", objective_ );
  adv.saveAttribute( "equalityConstraint_", equalityConstraint_ );
  adv.saveAttribute( "inequalityConstraint_", inequalityConstraint_ );
  adv.saveAttribute( "bounds_", bounds_ );
  adv.saveAttribute( "minimizationCollection_", minimizationCollection_ );
  adv.saveAttribute( "dimension_", dimension_ );
}

/* Method load() reloads the object from the StorageManager */
void OptimizationProblemImplementation::load(Advocate & adv)
{
  PersistentObject::load(adv);
  adv.loadAttribute( "objective_", objective_ );
  adv.loadAttribute( "equalityConstraint_", equalityConstraint_ );
  adv.loadAttribute( "inequalityConstraint_", inequalityConstraint_ );
  adv.loadAttribute( "bounds_", bounds_ );
  if (adv.hasAttribute("minimizationCollection_"))
    adv.loadAttribute("minimizationCollection_", minimizationCollection_);
  else
  {
    Bool minimization = true;
    adv.loadAttribute("minimization_", minimization);
    minimizationCollection_ = BoolCollection(1, minimization);
  }
  adv.loadAttribute( "dimension_", dimension_ );
}

END_NAMESPACE_OPENTURNS
